/**
 * \file: grl_configuration_devconf.c
 *
 * Common configuration interface for the graphic. This I/F is mapped in this C-file to the DEVCONF and could
 * be mapped to any other configuration mechanism depending on the product.
 *
 * \component: svg_common
 *
 * \author: CSpetzler
 *
 * \copyright: (c) 2003 - 2010 ADIT Corporation
 *
 * \history
 *
 */
#include "grl_configuration.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <svg_common.h>

#define FILE_NAME "svg_config.ini"
#define MAX_STRING 420

static GRL_error GRL_get_config_string(SVGChar * p_key, SVGChar * p_value, SVGUint32 size);

/**
 * \func GRL_get_hex32_config
 *
 * Get an unsigned 32 bit configuration value. If the value is stored as string a hexadecimal notation is assumed.
 *
 * \param  p_name                    : key value which is a string of up to 16 characters
 * \param  p_value                   : integer value for the key
 *
 * \return error of process or if positive number of elements in p_value (in this function always 1)
 *
 */
GRL_error GRL_get_hex32_config(SVGChar * p_name, SVGUint32 * p_value)
{
	GRL_error 	return_val = GRL_NO_ERROR;
	if (p_value == NULL)
	{
		SVG_COM_E("Null pointer as parameter in GRL_get_hex32_config");
		return_val = SVG_POINTER_NULL;
	}
	else
	{
		return_val = GRL_get_int_config(p_name, (SVGInt32*)p_value, 1);
		/* no error*/
		if (return_val > GRL_NO_ERROR)
		{
			/*only one element is returned*/
			return_val = 1;
		}
	}
	return return_val;
}

/**
 * \func GRL_get_int_config
 *
 * Get an integer configuration value.
 * If size = 0 the actual number of the elements for the key could be got.
 *
 * \param  p_name                    : key value which is a string of up to 16 characters
 * \param  p_value                   : integer value for the key
 * \param  size                      : number of the maximal elements to be stored in p_value
 *
 * \return error of process or if positiv number of elements in p_value
 *
 */
GRL_error GRL_get_int_config(SVGChar * p_name, SVGInt32 * p_value, SVGUint32 size)
{

	GRL_error 	return_val = GRL_NO_ERROR;
	SVGChar 	p_tmp[MAX_STRING];
	SVGChar * 	p_token		= NULL;
	SVGUint8 	loop		= 0;


	if ((p_value == NULL) && (size != 0))
	{
		SVG_COM_E("Nullpointer as parameter in GRL_get_int_config");
        return_val = SVG_POINTER_NULL;
	}
	else
	{
		return_val = GRL_get_config_string(p_name, p_tmp, MAX_STRING);
		if (return_val == GRL_NO_ERROR)
		{
			/*post process the returned string -> check # elements*/
			p_token = strtok (p_tmp,",");
			while (p_token != NULL)
			{
				/*store the string*/
				if ((loop < size) && (NULL != p_value))
				{
					p_value[loop] = (SVGInt32)strtoul(p_token,NULL,0);
				}
				loop++;
				p_token = strtok (NULL, ",");
			}
			return_val = loop;
		}
	}
	return return_val;
}

/**
 * \func GRL_get_string_config
 *
 * Get a string configuration value with a specific length.
 * If size = 0 the actual size of the string could be got.
 *
 * \param  p_name                    : key value which is a string of up to 16 characters
 * \param  p_value                   : value as array storing the character string
 * \param  size                      : number of the maximal elements to be stored in p_value (in byte)
 *
 * \return error of process or if positiv numeric length of the returning value.
 *
 */
GRL_error GRL_get_string_config(SVGChar * p_name, SVGUint8 * p_value, SVGUint32 size)
{

	GRL_error return_val = GRL_NO_ERROR;
	if ((p_value == NULL) && (size != 0))
	{
		SVG_COM_E("Null pointer as parameter in GRL_get_string_config");
		return_val = SVG_POINTER_NULL;
	}
	else
	{
		SVGChar 	p_tmp[MAX_STRING];
		if (size == 0)
		{
			return_val = GRL_get_config_string(p_name, (SVGChar *)p_tmp, MAX_STRING);
			if (return_val == GRL_NO_ERROR)
			{
				/*post process the returned string -> check length*/
				return_val = strlen((SVGChar *)p_tmp);
			}
		}
		else
		{
			return_val = GRL_get_config_string(p_name, (SVGChar *)p_value, size);
			if (return_val == GRL_NO_ERROR)
			{
				/*post process the returned string -> check length*/
				return_val = 0;
				if(NULL != p_value)
				{
					return_val = strlen((SVGChar *)p_value);
				}
			}
		}		
	}
	return return_val;
}

#ifdef USER_SPACE
static GRL_error GRL_get_config_string(SVGChar * p_key, SVGChar * p_value, SVGUint32 size)
{
	GRL_error 	return_val 	= GRL_NO_ERROR;
	FILE 	*	file		= NULL;
	SVGBoolean  found		= SVG_FALSE;

	file = fopen("/etc/"FILE_NAME, "r");

	if(file == NULL)
	{
		/*try local file*/
		file = fopen(FILE_NAME, "r");
		if(file == NULL)
		{
			return_val	= 	SVG_NOT_EXISTENT;
			SVG_COM_E("Couldn't open configuration file");
		}
	}

	if (return_val == GRL_NO_ERROR)
	{
		SVGChar p_temp[MAX_STRING];
		SVGChar * p_string	= NULL;
		SVGInt32	line	= 	-1;

		if(NULL != file)
		{
			while ( (fgets(p_temp, MAX_STRING, file)) && (found == SVG_FALSE))
			{
				line++;

				/*check for comment as first char*/
				if (p_temp[0] == '#')
				{
					continue;
				}

				p_temp[strlen(p_temp)-1] = '\0';

				p_string = strstr(p_temp, p_key);

				if ((NULL == p_string) || (p_string != p_temp))
				{
					/*Key not found at start of string*/
					continue;
				}

				if (*(p_string + strlen(p_key)) != '=')
				{
					/*Key isn't exactly same in the file or isn't followed by an equals sign*/
					continue;
				}

				/*Move to after the equals sign*/
				p_string += strlen(p_key) + 1;
				/*Return string*/
				if(NULL != p_value)
				{
					strncpy(p_value, p_string, size);
				}
				found = SVG_TRUE;
			}
		}
	}
	if (found == SVG_FALSE)
	{
		SVG_COM_E("Failed to read config entry %s",p_key);
		return_val	= 	SVG_NOT_EXISTENT;
	}
	if (file != NULL)
	{
		fclose(file);
	}
	return return_val;
}
#else //USER_SPACE

static GRL_error GRL_get_config_string(SVGChar * p_key, SVGChar * p_value, SVGUint32 size)
{
	/*no kernel version planned*/
	DISABLE_UNREFERENCED_FOR_LINT(p_key);
	DISABLE_UNREFERENCED_FOR_LINT(p_value);
	return SVG_NOT_EXISTENT;
}


#	include <linux/module.h>

	EXPORT_SYMBOL(GRL_get_string_config);
	EXPORT_SYMBOL(GRL_get_int_config);
	EXPORT_SYMBOL(GRL_get_hex32_config);
#endif //USER_SPACE
